jQuery(document).ready(function($) {
    let currentStoryIndex = 0;
    let stories = [];
    let storyDuration = 5000; // 5 seconds per story
    let progressInterval;
    let currentProgress = 0;

    function initStoryViewer() {
        $('.sc-story-item').on('click', function() {
            const storyId = $(this).data('story-id');
            loadStory(storyId);
        });

        $(document).on('keyup', function(e) {
            if (e.key === 'Escape') {
                closeStoryViewer();
            } else if (e.key === 'ArrowLeft') {
                showPreviousStory();
            } else if (e.key === 'ArrowRight') {
                showNextStory();
            }
        });
    }

    function loadStory(storyId) {
        $.ajax({
            url: scStoryData.ajaxUrl,
            type: 'POST',
            data: {
                action: 'load_story',
                story_id: storyId,
                nonce: scStoryData.nonce
            },
            success: function(response) {
                if (response.success) {
                    showStory(response.data);
                    markStoryViewed(storyId);
                }
            },
            error: function(xhr, status, error) {
                console.error('Error loading story:', error);
            }
        });
    }

    function showStory(storyData) {
        const viewer = createStoryViewer();
        viewer.addClass('active');
        
        updateStoryContent(storyData);
        startProgressBar();
    }

    function createStoryViewer() {
        if ($('.sc-story-viewer').length === 0) {
            const viewer = $(`
                <div class="sc-story-viewer">
                    <div class="sc-story-viewer-content">
                        <div class="sc-story-header">
                            <div class="sc-story-progress-container">
                                <div class="sc-story-progress">
                                    <div class="sc-story-progress-bar"></div>
                                </div>
                            </div>
                            <div class="sc-story-user-info">
                                <img src="" alt="User Avatar">
                                <span class="sc-story-username"></span>
                            </div>
                        </div>
                        <div class="sc-story-close">&times;</div>
                        <div class="sc-story-nav sc-story-prev">&lt;</div>
                        <div class="sc-story-nav sc-story-next">&gt;</div>
                        <div class="sc-story-media"></div>
                    </div>
                </div>
            `);
            
            viewer.find('.sc-story-close').on('click', closeStoryViewer);
            viewer.find('.sc-story-prev').on('click', showPreviousStory);
            viewer.find('.sc-story-next').on('click', showNextStory);
            
            $('body').append(viewer);
        }
        return $('.sc-story-viewer');
    }

    function updateStoryContent(storyData) {
        if (storyData.video_url) {
            $('.sc-story-media').html(`
                <video autoplay controls>
                    <source src="${storyData.video_url}" type="video/mp4">
                </video>
            `);
        } else if (storyData.thumbnail) {
            $('.sc-story-media').html(`
                <img src="${storyData.thumbnail}" alt="${storyData.title}">
            `);
        }
        
        $('.sc-story-username').text(storyData.title);
    }

    function startProgressBar() {
        const progressBar = $('.sc-story-progress-bar');
        currentProgress = 0;
        
        clearInterval(progressInterval);
        progressBar.css('width', '0%');
        
        progressInterval = setInterval(() => {
            currentProgress += (100 / (storyDuration / 100));
            progressBar.css('width', currentProgress + '%');
            
            if (currentProgress >= 100) {
                clearInterval(progressInterval);
                showNextStory();
            }
        }, 100);
    }

    function showPreviousStory() {
        // Implement previous story logic
    }

    function showNextStory() {
        // Implement next story logic
        closeStoryViewer();
    }

    function closeStoryViewer() {
        clearInterval(progressInterval);
        $('.sc-story-viewer').removeClass('active');
    }

    function markStoryViewed(storyId) {
        const storyItem = $(`.sc-story-item[data-story-id="${storyId}"]`);
        storyItem.removeClass('unviewed');
        
        $.ajax({
            url: '/wp-json/sc-story/v1/view-story',
            type: 'POST',
            data: {
                story_id: storyId
            },
            beforeSend: function(xhr) {
                xhr.setRequestHeader('X-WP-Nonce', scStoryData.nonce);
            }
        });
    }

    // Initialize event countdown timers
    function initEventTimers() {
        $('.sc-story-event-timer').each(function() {
            const eventDate = new Date($(this).data('event-date'));
            const timer = $(this);
            
            function updateTimer() {
                const now = new Date();
                const diff = eventDate - now;
                
                if (diff > 0) {
                    const days = Math.floor(diff / (1000 * 60 * 60 * 24));
                    const hours = Math.floor((diff % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
                    const minutes = Math.floor((diff % (1000 * 60 * 60)) / (1000 * 60));
                    
                    timer.html(`Event in: ${days}d ${hours}h ${minutes}m`);
                } else {
                    timer.html('Event started');
                }
            }
            
            updateTimer();
            setInterval(updateTimer, 60000); // Update every minute
        });
    }

    // Initialize everything
    initStoryViewer();
    initEventTimers();
});