<?php
/**
 * Plugin Name: SC Story
 * Plugin URI: https://suncode.ir
 * Description: Instagram-style story system for WordPress
 * Version: 1.0.0
 * Author: Sun Code Team
 * Author URI: https://suncode.ir
 * Text Domain: scstory
 * Domain Path: /languages
 */

if (!defined('ABSPATH')) exit;
include_once "metaboxes.php";
class SC_Story {
    private static $instance = null;

    public static function getInstance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    private function __construct() {
        $this->define_constants();
        $this->init_hooks();
    }

    private function define_constants() {
        define('SC_STORY_VERSION', '1.0.0');
        define('SC_STORY_PLUGIN_DIR', plugin_dir_path(__FILE__));
        define('SC_STORY_PLUGIN_URL', plugin_dir_url(__FILE__));
    }

    private function init_hooks() {
        add_action('init', array($this, 'register_post_type'));
        add_action('init', array($this, 'register_taxonomies'));
        add_action('admin_menu', array($this, 'add_admin_menu'));
        //add_action('add_meta_boxes', array($this, 'add_meta_boxes'));
        add_action('save_post', array($this, 'save_story_meta'));
        add_action('elementor/widgets/register', array($this, 'register_elementor_widget'));
        add_action('wp_enqueue_scripts', array($this, 'enqueue_scripts'));
        add_action('wp_ajax_load_story', array($this, 'ajax_load_story'));
        add_action('wp_ajax_nopriv_load_story', array($this, 'ajax_load_story'));
        add_action('rest_api_init', array($this, 'register_rest_routes'));
    }

    public function ajax_load_story() {
        check_ajax_referer('sc_story_nonce', 'nonce');
        
        $story_id = isset($_POST['story_id']) ? intval($_POST['story_id']) : 0;
        
        if (!$story_id) {
            wp_send_json_error('Invalid story ID');
        }
        
        $story = get_post($story_id);
        if (!$story) {
            wp_send_json_error('Story not found');
        }
        
        $video_url = get_post_meta($story_id, '_story_video_url', true);
        $related_link = get_post_meta($story_id, '_story_related_link', true);
        
        $response = array(
            'title' => $story->post_title,
            'content' => wpautop($story->post_content),
            'video_url' => $video_url,
            'related_link' => $related_link,
            'thumbnail' => get_the_post_thumbnail_url($story_id, 'full')
        );
        
        wp_send_json_success($response);
    }

    public function register_elementor_widget($widgets_manager) {
        // Include the widget class file
        require_once SC_STORY_PLUGIN_DIR . 'includes/elementor-widget.php';
        
        // Register the widget
        $widgets_manager->register(new \SC_Story\Widgets\Story_Widget());
    }

    public function add_admin_menu() {
        add_submenu_page(
            'studiare_panel',
            __('SC Story', 'studiare-core'),
            __('SC Story', 'studiare-core'),
            'manage_options',
            'sc-story',
            array($this, 'render_admin_page'),
            //'dashicons-format-gallery',
            20
        );
    }

    public function render_admin_page() {
        ?>
        <div class="wrap">
            <h1><?php echo esc_html(get_admin_page_title()); ?></h1>
            <div class="welcome-panel" style=" background: linear-gradient(115deg, rgb(249, 206, 52), rgb(238, 42, 123), rgb(98, 40, 215)); border-radius: 8px; padding: 15px; color: #fff; ">
                <div class="welcome-panel-content">
                    <h2 style="color: #fff; font-variation-settings: 'wght' 800;"><?php _e('Welcome to SC Story!', 'studiare-core'); ?></h2>
                    <p class="about-description"><?php _e('Create and manage your Instagram-style stories right from WordPress.', 'studiare-core'); ?></p>
                    <div class="welcome-panel-column-container">
                        <div class="welcome-panel-column">
                            <h3><?php _e('Get Started', 'studiare-core'); ?></h3>
                            <a class="button button-primary button-hero" href="<?php echo admin_url('post-new.php?post_type=sc_story'); ?>">
                                <?php _e('Create Your First Story', 'studiare-core'); ?>
                            </a>
                        </div>
                        <div class="welcome-panel-column">
                            <h3><?php _e('Quick Links', 'studiare-core'); ?></h3>
                            <ul>
                                <li><a href="<?php echo admin_url('edit.php?post_type=sc_story'); ?>"><?php _e('View All Stories', 'studiare-core'); ?></a></li>
                                <li><a href="<?php echo admin_url('edit-tags.php?taxonomy=story_category&post_type=sc_story'); ?>"><?php _e('Manage Categories', 'studiare-core'); ?></a></li>
                                <li><a href="<?php echo admin_url('edit-tags.php?taxonomy=story_tag&post_type=sc_story'); ?>"><?php _e('Manage Tags', 'studiare-core'); ?></a></li>
                            </ul>
                        </div>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }

    public function register_post_type() {
        $labels = array(
            'name' => __('Stories', 'studiare-core'),
            'singular_name' => __('Story', 'studiare-core'),
            'add_new' => __('Add New Story', 'studiare-core'),
            'add_new_item' => __('Add New Story', 'studiare-core'),
            'edit_item' => __('Edit Story', 'studiare-core'),
            'new_item' => __('New Story', 'studiare-core'),
            'view_item' => __('View Story', 'studiare-core'),
            'search_items' => __('Search Stories', 'studiare-core'),
            'not_found' => __('No stories found', 'studiare-core'),
            'not_found_in_trash' => __('No stories found in Trash', 'studiare-core'),
        );

        $args = array(
            'labels' => $labels,
            'public' => true,
            'has_archive' => true,
            'menu_icon' => 'dashicons-format-gallery',
            'supports' => array('title','thumbnail'), //'editor', 'comments'
            'show_in_rest' => true,
            'exclude_from_search' => true,
			'publicly_queryable'  => false,
            'show_ui'            => true,
            //'show_in_menu' => 'sc-story',
        );

        register_post_type('sc_story', $args);
    }

    public function register_taxonomies() {
        // Categories
        $cat_labels = array(
            'name' => __('Story Categories', 'studiare-core'),
            'singular_name' => __('Story Category', 'studiare-core'),
        );

        register_taxonomy('story_category', 'sc_story', array(
            'labels' => $cat_labels,
            'hierarchical' => true,
            'show_in_rest' => true,
        ));

        // Tags
        $tag_labels = array(
            'name' => __('Story Tags', 'studiare-core'),
            'singular_name' => __('Story Tag', 'studiare-core'),
        );

        register_taxonomy('story_tag', 'sc_story', array(
            'labels' => $tag_labels,
            'hierarchical' => true,
            'show_in_rest' => true,
        ));
    }

    public function add_meta_boxes() {
        add_meta_box(
            'story_details',
            __('Story Details', 'studiare-core'),
            array($this, 'render_story_meta_box'),
            'sc_story',
            'normal',
            'high'
        );
    }

    public function render_story_meta_box($post) {
        wp_nonce_field('story_meta_box', 'story_meta_box_nonce');
        
        $video_url = get_post_meta($post->ID, '_story_video_url', true);
        $related_link = get_post_meta($post->ID, '_story_related_link', true);
        $event_date = get_post_meta($post->ID, '_story_event_date', true);
        ?>
        <div class="story-meta-box">
            <p>
                <label for="story_video_url"><?php _e('Video URL:', 'studiare-core'); ?></label>
                <input type="url" id="story_video_url" name="story_video_url" value="<?php echo esc_url($video_url); ?>" class="widefat">
            </p>
            <p>
                <label for="story_related_link"><?php _e('Related Link:', 'studiare-core'); ?></label>
                <input type="url" id="story_related_link" name="story_related_link" value="<?php echo esc_url($related_link); ?>" class="widefat">
            </p>
            <p>
                <label for="story_event_date"><?php _e('Event Date:', 'studiare-core'); ?></label>
                <input type="datetime-local" id="story_event_date" name="story_event_date" value="<?php echo esc_attr($event_date); ?>" class="widefat">
            </p>
        </div>
        <?php
    }

    public function save_story_meta($post_id) {
        if (!isset($_POST['story_meta_box_nonce']) || 
            !wp_verify_nonce($_POST['story_meta_box_nonce'], 'story_meta_box')) {
            return;
        }

        if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) {
            return;
        }

        if (!current_user_can('edit_post', $post_id)) {
            return;
        }

        if (isset($_POST['story_video_url'])) {
            update_post_meta($post_id, '_story_video_url', sanitize_url($_POST['story_video_url']));
        }

        if (isset($_POST['story_related_link'])) {
            update_post_meta($post_id, '_story_related_link', sanitize_url($_POST['story_related_link']));
        }

        if (isset($_POST['story_event_date'])) {
            update_post_meta($post_id, '_story_event_date', sanitize_text_field($_POST['story_event_date']));
        }
    }

    public function enqueue_scripts() {
        /*
        wp_enqueue_style(
            'sc-story-style',
            SC_STORY_PLUGIN_URL . 'assets/css/style.css',
            array(),
            SC_STORY_VERSION
        );

        wp_enqueue_script(
            'sc-story-script',
            SC_STORY_PLUGIN_URL . 'assets/js/script.js',
            array('jquery'),
            SC_STORY_VERSION,
            true
        );
        

        wp_localize_script('sc-story-script', 'scStoryData', array(
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('sc_story_nonce'),
        ));
        */
    }

    public function register_rest_routes() {
        register_rest_route('sc-story/v1', '/view-story', array(
            'methods' => 'POST',
            'callback' => array($this, 'mark_story_viewed'),
            'permission_callback' => '__return_true',
        ));
    }

    public function mark_story_viewed($request) {
        $story_id = $request->get_param('story_id');
        $user_id = get_current_user_id();
        
        if (!$user_id) {
            return new WP_Error('unauthorized', 'User must be logged in', array('status' => 401));
        }

        $viewed_stories = get_user_meta($user_id, '_viewed_stories', true);
        if (!is_array($viewed_stories)) {
            $viewed_stories = array();
        }

        $viewed_stories[] = $story_id;
        update_user_meta($user_id, '_viewed_stories', array_unique($viewed_stories));

        return rest_ensure_response(array('success' => true));
    }
}

// Initialize the plugin
SC_Story::getInstance();