<?php
// --- Helpers: truthy normalizer (handles 'on', '1', 1, true, 'yes') ---
if (!function_exists('studiare_truthy')) {
  function studiare_truthy($v) {
    return in_array($v, array('on', '1', 1, true, 'yes', 'true'), true);
  }
}

// --- Remove this if you still have the old add_filter('the_content', ...) ---
// We no longer inject lessons into content; we print via hook outside the content.
// remove_filter('the_content', 'add_shortcode_to_end_of_content'); // <- ensure removed if exists

// --- Keep the outside-content hook (prints when NOT using tabs) ---
add_action('studiare_after_product_content_holder', function () {
  // Only for single product main query
  if (!is_singular('product') || !is_main_query()) return;

  $post_id = get_the_ID();
  if (!$post_id) return;

  $prefix = '_studiare_';
  $view_lessons_end_of_content = get_post_meta($post_id, $prefix . 'view_lessons_end_of_content', true);

  // If per-product toggle is off, bail
  if (!studiare_truthy($view_lessons_end_of_content)) return;

  // If global setting says "display in tabs", DO NOT print here
  $lessons_display_in_tab = function_exists('codebean_option') ? codebean_option('sc_lessons_display_in_tab') : false;
  if (studiare_truthy($lessons_display_in_tab)) return;

  echo '<div class="studi-lessons-after-content product-single-content" aria-label="' . esc_attr__('Lessons', 'studiare') . '">';
  echo do_shortcode('[sc_get_lessons_for_product]');
  echo '</div>';
}, 10);

// --- Robust WooCommerce Tabs injection ---
function studiare_add_lessons_tab_to_woocommerce_tabs($tabs) {
  // Get product ID safely (works even if global $post is not set properly)
  $product_id = 0;

  if (function_exists('wc_get_product')) {
    $product_obj = wc_get_product();
    if ($product_obj) $product_id = $product_obj->get_id();
  }

  if (!$product_id) {
    // Fallback to get_the_ID() for themes loading tabs inside the main product loop
    $product_id = get_the_ID();
  }

  if (!$product_id) {
    // If we still don't have an ID, do not modify tabs
    return $tabs;
  }

  // Per-product toggle must be ON
  $prefix = '_studiare_';
  $view_lessons_end_of_content = get_post_meta($product_id, $prefix . 'view_lessons_end_of_content', true);
  if (!studiare_truthy($view_lessons_end_of_content)) {
    return $tabs;
  }

  // Global: show lessons in a tab?
  $lessons_display_in_tab = function_exists('codebean_option') ? codebean_option('sc_lessons_display_in_tab') : false;
  if (!studiare_truthy($lessons_display_in_tab)) {
    // If not in tabs mode, leave it for the outside-content hook
    return $tabs;
  }

  // Add/override lessons tab
  $tabs['lessons_tab'] = array(
    'title'    => esc_html__('Lessons', 'studiare'),
    'priority' => 10, // adjust if you want it earlier/later than Description/Reviews
    'callback' => 'studiare_render_lessons_tab_content',
  );

  return $tabs;
}
// Priority set to 50 to run after most themes/plugins have built their tabs array
add_filter('woocommerce_product_tabs', 'studiare_add_lessons_tab_to_woocommerce_tabs', 50);

// --- Tab renderer ---
function studiare_render_lessons_tab_content() {
  echo do_shortcode('[sc_get_lessons_for_product]');
}


// Render the content of the lessons tab
function render_lessons_tab_content() {
    $shortcode = '[sc_get_lessons_for_product]';
    echo do_shortcode($shortcode);
}

// Register shortcode
add_shortcode("sc_get_lessons_for_product", "sc_get_lessons_for_product");

// Check if user has passed the required quizzes
function has_passed_previous_quizzes($user_id, $quiz_ids) {
    global $wpdb;
    if (empty($quiz_ids)) {
        return true;
    }

    if (!$user_id) {
        return false;
    }

    $table_name = $wpdb->prefix . 'studiare_quiz_results';
    foreach ($quiz_ids as $quiz_id) {
        $result = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT details FROM $table_name WHERE user_id = %d AND quiz_id = %d ORDER BY timestamp DESC LIMIT 1",
                $user_id,
                $quiz_id
            )
        );

        if (!$result) {
            return false;
        }

        $details = maybe_unserialize($result->details);
        $details = is_array($details) ? $details : [];
        $status = isset($details['status']) ? $details['status'] : 'unknown';

        if ($status !== 'passed') {
            return false;
        }
    }
    return true;
}

// Get the title of the last failed quiz
function get_last_failed_quiz_title($user_id, $quiz_ids) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'studiare_quiz_results';
    foreach ($quiz_ids as $quiz_id) {
        $result = $wpdb->get_row(
            $wpdb->prepare(
                "SELECT details FROM $table_name WHERE user_id = %d AND quiz_id = %d ORDER BY timestamp DESC LIMIT 1",
                $user_id,
                $quiz_id
            )
        );
        if (!$result || (isset($result->details) && maybe_unserialize($result->details)['status'] !== 'passed')) {
            $quiz_post = get_post($quiz_id);
            return $quiz_post ? $quiz_post->post_title : __('Previous Quiz', 'studiare');
        }
    }
    return __('Previous Quiz', 'studiare');
}

function render_lesson_media($lesson_video, $lesson_audio) {
    $output = "<div class='sc_le_vi_au'>";
    $enable_suncode_player = false;
    if (class_exists('Redux') && function_exists('codebean_option')) {
        $enable_suncode_player = in_array(codebean_option('enable_suncode_player'), ['1', 1, true], true);
    }

    $controls_attr = '';
    if (!$enable_suncode_player) {
        $controls_attr = ' controls';
    }

    if (!empty($lesson_video)) {
        $video_type = str_ends_with($lesson_video, '.m3u8') ? 'application/x-mpegURL' : 'video/mp4';
        $unique_id = 'video_' . wp_generate_uuid4();
        $output .= "<br>";
        $output .= "<video class='studi_lesson_video scplayer scplayer--full-ui scplayer--video scplayer--html5' preload='none' style='width:100%'$controls_attr controlsList='nodownload' oncontextmenu='return false;' data-id='$unique_id'>";
        $output .= "<source src='" . esc_url($lesson_video) . "' type='" . esc_attr($video_type) . "'>";
        $output .= "</video>";
        $output .= "<br>";
    }

    if (!empty($lesson_audio)) {
        $unique_id = 'audio_' . wp_generate_uuid4();
        $output .= "<audio class='studi_lesson_audio scplayer scplayer--full-ui scplayer--audio scplayer--html5' preload='none' style='width:100%'$controls_attr controlsList='nodownload' oncontextmenu='return false;' data-id='$unique_id'>";
        $output .= "<source src='" . esc_url($lesson_audio) . "' type='audio/mpeg'>";
        $output .= "</audio>";
        $output .= "<br>";
    }

    $output .= "</div>";
    return $output;
}

function sc_get_lessons_for_product($post_id = null) {
    global $wpdb;
    ob_start();

    // Set post_id to current post if not provided
    if (empty($post_id)) {
        $post_id = get_the_ID();
    }

    // Validate post_id
    if (!is_numeric($post_id) || !$post_id) {
        return '<p>Error: Invalid post ID.</p>';
    }

    // Get lessons_group meta
    $lessons_group = get_post_meta($post_id, 'lessons_group', true);

    // Check if lessons_group is valid
    if (!is_array($lessons_group) || empty($lessons_group)) {
        return '<p>' . esc_html__('No lessons or quizzes found.', 'studiare') . '</p>';
    }

    // Load quiz settings
    $quiz_settings_override = get_post_meta($post_id, '_studiare_sc_quiz_settings_override', true) ?: 'default_from_theme_options';
    if ($quiz_settings_override === 'override') {
        $lock_lessons_based_on_quiz = get_post_meta($post_id, '_studiare_sc_lock_lessons_based_on_quiz', true) === 'on';
        $lock_scope = get_post_meta($post_id, '_studiare_sc_lock_lessons_scope', true) ?: 'all_items';
        $always_allow_first_item = get_post_meta($post_id, '_studiare_sc_always_allow_first_item', true) === 'on';
    } else {
        $lock_lessons_based_on_quiz = codebean_option('sc_lock_lessons_based_on_quiz') ?? false;
        $lock_scope = codebean_option('sc_lock_lessons_scope') ?? 'all_items';
        $always_allow_first_item = codebean_option('sc_always_allow_first_item') ?? true;
    }

    // Track quizzes and their positions
    $global_quiz_positions = [];
    $global_index = 0;

    // Collect all quiz positions
    foreach ($lessons_group as $section) {
        $lessons = isset($section['lessons_list']) ? $section['lessons_list'] : [];
        if (is_array($lessons) && !empty($lessons)) {
            foreach ($lessons as $lesson) {
                if (is_numeric($lesson) && ($post = get_post($lesson)) && get_post_type($lesson) === 'sc-quiz' && get_option('studiare_quiz_post_type_enabled', true)) {
                    $global_quiz_positions[] = $global_index;
                }
                $global_index++;
            }
        }
    }

    $global_index = 0; // Reset for rendering
    $global_previous_quiz_ids = [];

    foreach ($lessons_group as $section) {
        $title = isset($section['title']) ? $section['title'] : '';
        $list_open_situ = isset($section['list_open_situ']) ? $section['list_open_situ'] : null;
        $section_number = isset($section['section_number']) ? $section['section_number'] : '';
        $section_num_title = isset($section['section_num_title']) ? $section['section_num_title'] : '';
        $lessons = isset($section['lessons_list']) ? $section['lessons_list'] : [];
        $showicon = isset($section['showicon']) ? $section['showicon'] : null;

        $lesson_number = 1;
        $section_previous_quiz_ids = ($lock_scope === 'within_section') ? [] : $global_previous_quiz_ids;
        $section_index = 0;
        $section_has_quiz = false;

        $css_class = "course-section";
        $rand_id = rand();
        $openclass = ($list_open_situ == 'on') ? " active_tab_by_suncode" : " ";
        ?>
        <div class="<?php echo esc_attr($css_class); ?>">
            <?php if (!empty($title)) : ?>
                <div data-id="<?php echo $rand_id; ?>" class="sc-course-lesson-toggle-wrapper">
                    <h5 class="course-section-title">
                        <?php if (!empty($section_num_title) || !empty($section_number)) { ?>
                            <span class="chapterholder">
                                <?php if (!empty($section_num_title)) : ?><span class="scchapturetitle"><?php echo esc_html($section_num_title); ?></span><?php endif; ?>
                                <?php if (!empty($section_number)) : ?><span class="scchapturenumber"><?php echo esc_html($section_number); ?></span><?php endif; ?>
                            </span>
                        <?php } ?>
                        <?php if ($showicon == 'on') : ?><i class="fal fa-ellipsis-h-alt"></i><?php endif; ?> <?php echo esc_html($title); ?>
                    </h5>
                    <div class="sc-course-lesson-toggle"><i class="fad fa-chevron-down"></i></div>
                </div>
            <?php endif; ?>
            <div class="panel-group" style="display:none;">
                <?php
                if (is_array($lessons) && !empty($lessons)) {
                    $current_user = wp_get_current_user();
                    $user_id = $current_user->ID;
                    $bought_course = false;

                    // Check if user has purchased the course
                    if (!empty($current_user->user_email) && !empty($user_id)) {
                        if (wc_customer_bought_product($current_user->user_email, $user_id, get_the_ID())) {
                            $bought_course = true;
                        }
                    }
                    $isbought = studi_has_bought_items($user_id, get_the_ID());
                    if ($isbought == "true") {
                        $bought_course = true;
                    }
                    
                    
                    // Check if user has an active subscription using Studiare_Subscription_Manager
                    $has_active_subscription = false;
                    if (class_exists('Studiare_Subscription_Manager') && $user_id) {
                        $subscriptions = Studiare_Subscription_Manager::get_user_subscriptions($user_id);
                        if (!empty($subscriptions)) {
                            $bought_course = true;
                            $has_active_subscription = true;
                        }
                    }

                    foreach ($lessons as $lesson) {
                        if (!is_numeric($lesson) || !($post = get_post($lesson))) {
                            $global_index++;
                            $section_index++;
                            continue;
                        }

                        $post_type = get_post_type($lesson);
                        if ($post_type === 'sc-quiz' && !get_option('studiare_quiz_post_type_enabled', true)) {
                            $global_index++;
                            $section_index++;
                            continue;
                        }

                        $is_first_quiz = false;
                        $is_current_quiz_blocking = false;
                        if ($post_type === 'sc-quiz') {
                            if (!$section_has_quiz && $lock_scope === 'within_section') {
                                $section_has_quiz = true;
                                $is_first_quiz = true;
                            }
                            $section_previous_quiz_ids[] = $lesson;
                            if ($lock_scope === 'all_items') {
                                $global_previous_quiz_ids[] = $lesson;
                            }

                            // Check if user failed this quiz
                            if ($user_id) {
                                $table_name = $wpdb->prefix . 'studiare_quiz_results';
                                $result = $wpdb->get_row(
                                    $wpdb->prepare(
                                        "SELECT details FROM $table_name WHERE user_id = %d AND quiz_id = %d ORDER BY timestamp DESC LIMIT 1",
                                        $user_id,
                                        $lesson
                                    )
                                );
                                if ($result) {
                                    $details = maybe_unserialize($result->details);
                                    $details = is_array($details) ? $details : [];
                                    $status = isset($details['status']) ? $details['status'] : 'unknown';
                                    if ($status !== 'passed') {
                                        $is_current_quiz_blocking = true;
                                    }
                                }
                            }
                        }

                        // Determine which quizzes block this item
                        $quiz_ids_to_check = [];
                        if ($lock_lessons_based_on_quiz) {
                            if ($lock_scope === 'within_section') {
                                $quiz_ids_to_check = $section_previous_quiz_ids;
                            } else {
                                foreach ($global_quiz_positions as $quiz_pos) {
                                    if ($quiz_pos < $global_index) {
                                        $quiz_ids_to_check = array_slice($global_previous_quiz_ids, 0, array_search($lesson, $global_previous_quiz_ids) ?: count($global_previous_quiz_ids));
                                    }
                                }
                            }
                        }

                        // Check if item is accessible
                        $is_access_blocked = false;
                        if ($lock_lessons_based_on_quiz) {
                            $is_access_blocked = !has_passed_previous_quizzes($user_id, $quiz_ids_to_check);
                        }

                        // Allow first item if configured
                        $is_first_item = false;
                        if ($always_allow_first_item) {
                            $is_first_item = ($lock_scope === 'within_section' ? $section_index === 0 : $global_index === 0);
                        }

                        // Allow first quiz and current blocking quiz
                        if ($is_first_quiz || $is_current_quiz_blocking) {
                            $is_access_blocked = false;
                        }

                        // Block access if needed
                        if ($lock_lessons_based_on_quiz && $is_access_blocked && !$is_first_item) {
                            $last_failed_quiz_title = get_last_failed_quiz_title($user_id, $quiz_ids_to_check);
                            ?>
                            <div class="course-lesson">
                                <div class="course-panel-heading">
                                    <div class="panel-heading-left">
                                        <div class="course-lesson-icon">
                                            <i class="fas fa-lock"></i>
                                        </div>
                                        <div class="title">
                                            <h4>
                                                <?php
                                                if ($post_type === 'sc-quiz') {
                                                    $quiz_title = !empty($post->post_title) ? $post->post_title : __('Quiz', 'studiare');
                                                    echo esc_html($quiz_title);
                                                    echo '<span class="badge-item quiz">' . __('Quiz', 'studiare') . '</span>';
                                                } else {
                                                    $item = get_post_meta($lesson, 'lesson_data', true);
                                                    $title = isset($item[0]['lesson_title']) ? $item[0]['lesson_title'] : null;
                                                    echo esc_html($title);
                                                }
                                                ?>
                                            </h4>
                                        </div>
                                    </div>
                                </div>
                                <div class="panel-content">
                                    <div class="panel-content-inner">
                                        <div class="lessonaccessdenied">
                                            <span class="icon"><i class="fas fa-exclamation-triangle"></i></span>
                                            <?php
                                            if (!$user_id) {
                                                $login_url = wp_login_url(get_permalink());
                                                printf(
                                                    esc_html__('You need to log in and pass the "%s" quiz to access this content.', 'studiare'),
                                                    esc_html($last_failed_quiz_title)
                                                );
                                            } else {
                                                printf(
                                                    esc_html__('You need to pass the "%s" quiz to access this content.', 'studiare'),
                                                    esc_html($last_failed_quiz_title)
                                                );
                                            }
                                            ?>
                                        </div>
                                    </div>
                                </div>
                            </div>
                            <?php
                            $global_index++;
                            $section_index++;
                            continue;
                        }

                        if ($post_type === 'sc-quiz') {
                            $quiz_title = !empty($post->post_title) ? $post->post_title : __('Quiz', 'studiare');
                            $restrict_to_members = get_post_meta($lesson, '_quiz_restrict_to_members', true);
                            $restrict_to_products = get_post_meta($lesson, '_quiz_restrict_to_products', true);
                            $widget_description = get_post_meta($lesson, '_quiz_widget_description', true);
                            $login_placeholder = codebean_option('login_lesson_message');
                            $current_product_id = get_the_ID();
                            $is_current_product_restricted = is_array($restrict_to_products) && in_array($current_product_id, $restrict_to_products);

                            // Get quiz status
                            $quiz_status = 'pending';
                            $quiz_status_class = 'pending';
                            if ($user_id) {
                                $table_name = $wpdb->prefix . 'studiare_quiz_results';
                                $result = $wpdb->get_row(
                                    $wpdb->prepare(
                                        "SELECT details FROM $table_name WHERE user_id = %d AND quiz_id = %d ORDER BY timestamp DESC LIMIT 1",
                                        $user_id,
                                        $lesson
                                    )
                                );
                                if ($result) {
                                    $details = maybe_unserialize($result->details);
                                    $details = is_array($details) ? $details : [];
                                    $quiz_status = isset($details['status']) ? $details['status'] : 'pending';
                                    $quiz_status_class = ($quiz_status === 'passed') ? 'passed' : 'failed';
                                }
                            }
                            ?>
                            <div class="course-lesson">
                                <div class="course-panel-heading scquiz">
                                    <div class="panel-heading-left">
                                        <div class="course-lesson-icon">
                                            <i class="fal fa-question-circle"></i>
                                        </div>
                                        <div class="title">
                                            <h4><?php echo esc_html($quiz_title); ?>
                                                <span class="badge-item quiz"><?php echo __('Quiz', 'studiare') ?></span>
                                                <?php if ($user_id && $quiz_status !== 'pending') : ?>
                                                    <span class="badge-item status-<?php echo esc_attr($quiz_status_class); ?>">
                                                        <?php
                                                        if ($quiz_status === 'passed') {
                                                            echo '<i class="fal fa-check-circle"></i> ' . __('Passed', 'studiare');
                                                        } else {
                                                            echo '<i class="fal fa-times-circle"></i> ' . __('Failed', 'studiare');
                                                        }
                                                        ?>
                                                    </span>
                                                <?php endif; ?>
                                            </h4>
                                            <?php if (!empty($widget_description)) : ?>
                                                <p class="subtitle"><?php echo esc_html($widget_description); ?></p>
                                            <?php endif; ?>
                                        </div>
                                    </div>
                                    <div class="panel-heading-right">
                                        <?php if ($restrict_to_members === 'yes') : ?>
                                            <div class="private-lesson<?php echo is_user_logged_in() ? ' privateunlock' : ''; ?>"><i class="fa fa-user-lock"></i><span><?php echo esc_html($login_placeholder ? $login_placeholder : __('for Users', 'studiare')); ?></span></div>
                                        <?php endif; ?>
                                        <?php if (is_array($restrict_to_products) && !empty($restrict_to_products)) : ?>
                                                <div class="private-lesson<?php echo $has_active_subscription ? ' privateunlock hassubscriptions' : ''; ?>"><i class="fa fa-lock"></i><span><?php esc_html_e('Private', 'studiare'); ?></span></div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <div class="panel-content scquiz">
                                    <div class="panel-content-inner">
                                        <?php
                                        if ($restrict_to_members === 'yes' && !is_user_logged_in()) {
                                            ?>
                                            <div class="lessonaccessdenied">
                                                <span class="icon"><i class="fas fa-user-lock"></i></span>
                                                <?php esc_html_e('This quiz is private. Please log in to access.', 'studiare'); ?>
                                            </div>
                                            <?php
                                        } elseif ($is_current_product_restricted && !$bought_course) {
                                            ?>
                                            <div class="lessonaccessdenied">
                                                <span class="icon"><i class="fas fa-exclamation-triangle"></i></span>
                                                <?php esc_html_e('This quiz is private, for full access to all lessons you need to buy this course.', 'studiare'); ?>
                                            </div>
                                            <?php
                                        } else {
                                            $quiz_shortcode_output = do_shortcode('[studiare_quiz pid="'.$current_product_id.'" id="' . esc_attr($lesson) . '"]');
                                            echo $quiz_shortcode_output ?: '<p>' . esc_html__('Quiz content not available.', 'studiare') . '</p>';
                                        }
                                        ?>
                                    </div>
                                </div>
                            </div>
                            <?php
                        } else {
                            $item = get_post_meta($lesson, 'lesson_data', true);
                            if (!is_array($item) || empty($item)) {
                                $global_index++;
                                $section_index++;
                                continue;
                            }

                            $icon = isset($item[0]['icon']) ? $item[0]['icon'] : null;
                            $title = isset($item[0]['lesson_title']) ? $item[0]['lesson_title'] : null;
                            $subtitle = isset($item[0]['lesson_subtitle']) ? $item[0]['lesson_subtitle'] : null;
                            $badge = isset($item[0]['badge']) ? $item[0]['badge'] : null;
                            $preview_video = isset($item[0]['preview_video']) ? $item[0]['preview_video'] : '';
                            $private_lesson = isset($item[0]['private_lesson']) ? $item[0]['private_lesson'] : null;
                            if ($private_lesson == "on") {
                                $private_lesson = "lesson_protected";
                            }
                            $sc_pls = isset($item[0]['sc_pls']) ? $item[0]['sc_pls'] : null;
                            $content_main = get_post_field('post_content', $lesson);
                            $content = wpautop($content_main);

                            $atts['sc_download_file'] = isset($item[0]['sc_download_file']) ? $item[0]['sc_download_file'] : '';
                            $atts['sc_download_access'] = isset($item[0]['sc_download_access']) ? $item[0]['sc_download_access'] : '';

                            $lesson_video = isset($item[0]['lesson_video']) ? $item[0]['lesson_video'] : null;
                            $lesson_audio = isset($item[0]['lesson_audio']) ? $item[0]['lesson_audio'] : null;

                            $lesson_number_auto = codebean_option('sc_lesson_number_auto') ?? false;
                            $login_toast_title = codebean_option('login_toast_title');
                            $login_toast_message = codebean_option('login_toast_message');
                            $bought_toast_title = codebean_option('bought_toast_title');
                            $bought_toast_message = codebean_option('bought_toast_message');
                            $private_placeholder = codebean_option('private_lesson_message');
                            $login_placeholder = codebean_option('login_lesson_message');
                            ?>
                            <div class="course-lesson">
                                <div class="course-panel-heading">
                                    <div class="panel-heading-left">
                                        <?php if ($lesson_number_auto || (!empty($icon) && !$lesson_number_auto)) { ?>
                                            <div class="course-lesson-icon">
                                                <?php
                                                if ($lesson_number_auto) {
                                                    echo '<span class="lesson-number">' . esc_html($lesson_number) . '</span>';
                                                } else {
                                                    $icon = substr_replace($icon, " ", 3, -strlen($icon));
                                                    echo "<i class='$icon'></i>";
                                                }
                                                ?>
                                            </div>
                                        <?php } ?>
                                        <?php if (!empty($title) || !empty($subtitle)) : ?>
                                            <div class="title">
                                                <h4><?php echo esc_html($title); ?>
                                                    <?php if (!empty($badge) && $badge != 'no_badge') : ?>
                                                        <span class="badge-item <?php echo esc_attr($badge); ?>"><?php echo esc_html(add_class_value_in_any_lang($badge)); ?></span>
                                                    <?php endif; ?>
                                                </h4>
                                                <?php if (!empty($subtitle)) : ?><p class="subtitle"><?php echo esc_html($subtitle); ?></p><?php endif; ?>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                    <?php
                                    $lesson_number++;
                                    ?>
                                    <?php if (!empty($preview_video) || !empty($private_lesson) || !empty($sc_pls)) : ?>
                                        <?php
                                        if (str_contains($preview_video, ".aparat.com/v")) {
                                            $course_video_aparat = substr($preview_video, strrpos($preview_video, '/') + 1);
                                            $preview_video = studi_get_aparat_file_link($course_video_aparat);
                                        }
                                        ?>
                                        <div class="panel-heading-right">
                                            <?php if (!empty($preview_video)) : ?>
                                                <a class="video-lesson-preview preview-button" href="<?php echo esc_url($preview_video); ?>"><i class="fa fa-play-circle"></i><?php esc_html_e('Preview', 'studiare'); ?></a>
                                            <?php endif; ?>
                                            <?php if ($private_lesson == 'lesson_protected') : ?>
                                                <div class="private-lesson<?php echo $has_active_subscription ? ' privateunlock hassubscriptions' : ''; ?>"><i class="fa fa-lock"></i><span><?php esc_html_e('Private', 'studiare'); ?></span></div>
                                            <?php elseif ($private_lesson == 'lesson_free_users') : ?>
                                                <div class="private-lesson<?php echo is_user_logged_in() ? ' privateunlock' : ''; ?>"><i class="fa fa-user-lock"></i><span><?php echo esc_html($login_placeholder ? $login_placeholder : __('for Users', 'studiare')); ?></span></div>
                                            <?php endif; ?>
                                            <?php
                                            $sc_protect_link_status = ($sc_pls == "on") ? 'true' : "false";
                                            if ($atts['sc_download_file'] !== 'none') {
                                                echo generate_dl_section($atts, get_the_ID(), $private_lesson, $bought_course, $login_toast_title, $login_toast_message, $bought_toast_title, $bought_toast_message, $sc_protect_link_status);
                                            }
                                            ?>
                                        </div>
                                    <?php endif; ?>
                                </div>
                                <div class="panel-content">
                                    <div class="panel-content-inner">
                                        <?php
                                        if ($private_lesson == 'lesson_protected') {
                                            if ($bought_course) {
                                                echo do_shortcode($content);
                                                   if (!empty($lesson_video) || !empty($lesson_audio)) {
                                                        echo render_lesson_media($lesson_video, $lesson_audio);
                                                    }
                                            } else {
                                                ?>
                                                <div class="lessonaccessdenied">
                                                    <span class="icon"><i class="fas fa-exclamation-triangle"></i></span>
                                                    <?php
                                                    if (!empty($private_placeholder)) {
                                                        echo wp_kses_post($private_placeholder);
                                                    } else {
                                                        esc_html_e('This lesson is private, for full access to all lessons you need to buy this course.', 'studiare');
                                                    }
                                                    ?>
                                                </div>
                                                <?php
                                            }
                                        } elseif ($private_lesson == 'lesson_free_users' && !is_user_logged_in()) {
                                            ?>
                                            <div class="lessonaccessdenied">
                                                <span class="icon"><i class="fas fa-user-lock"></i></span>
                                                <?php esc_html_e('This lesson is private. Please log in to access.', 'studiare'); ?>
                                            </div>
                                            <?php
                                        } else {
                                            echo do_shortcode($content);
                                            if (!empty($lesson_video) || !empty($lesson_audio)) {
                                                echo render_lesson_media($lesson_video, $lesson_audio);
                                            }
                                        }
                                        ?>
                                    </div>
                                </div>
                            </div>
                            <?php
                        }
                        $global_index++;
                        $section_index++;
                    }
                }
                ?>
            </div>
        </div>
        <script>
            jQuery(".sc-course-lesson-toggle-wrapper").off('click').click(function(el) {
                var id = jQuery(this).data('id');
                var query = "[data-id='" + id + "']";
                var item = jQuery(query);
                item.toggleClass("active_tab_by_suncode");
                item.next('.panel-group').slideToggle();
            });
            <?php if ($list_open_situ == 'on') { ?>
                jQuery(document).ready(function() {
                    var query = "[data-id='" + <?php echo $rand_id; ?> + "']";
                    var item = jQuery(query);
                    item.toggleClass("active_tab_by_suncode");
                    item.next('.panel-group').slideToggle();
                });
            <?php } ?>
        </script>
        <?php
    }
    return ob_get_clean();
}
?>