<?php

namespace Duplicator\Utils\Email;

use DUP_PRO_Global_Entity;
use DUP_PRO_Log;
use DUP_PRO_Package_Upload_Info;
use Duplicator\Core\Views\TplMng;
use Duplicator\Libs\Snap\SnapWP;
use Duplicator\Utils\CronUtils;
use Exception;

/**
 * Email summary bootstrap
 */
class EmailSummaryBootstrap
{
    const CRON_HOOK = 'duplicator_pro_email_summary_cron';

    /**
     * Init
     *
     * @return void
     */
    public static function init(): void
    {
        //Init Preview page
        \Duplicator\Controllers\EmailSummaryPreviewPageController::getInstance();

        //Storage hooks
        add_action('duplicator_pro_after_storage_create', function ($storageId): void {
            EmailSummary::getInstance()->addStorage($storageId);
        });
        add_action('duplicator_pro_after_storage_delete', function ($storageId): void {
            EmailSummary::getInstance()->removeStorage($storageId);
        });

        //Schedule hooks
        add_action('duplicator_pro_after_schedule_create', function ($schedule): void {
            EmailSummary::getInstance()->addSchedule($schedule);
        });
        add_action('duplicator_pro_after_schedule_delete', function ($scheduleId): void {
            EmailSummary::getInstance()->removeSchedule($scheduleId);
        });

        //Package hooks
        add_action('duplicator_pro_build_completed', function ($package): void {
            EmailSummary::getInstance()->addPackage($package);
        });
        add_action('duplicator_pro_build_fail', function ($package): void {
            EmailSummary::getInstance()->addFailed($package);
        });

        //Backup transfer hooks
        add_action('duplicator_pro_transfer_failed', function (DUP_PRO_Package_Upload_Info $uploadInfo): void {
            if ($uploadInfo->isDownloadFromRemote()) {
                return;
            }

            EmailSummary::getInstance()->addFailedUpload($uploadInfo);
        });

        add_action('duplicator_pro_transfer_cancelled', function (DUP_PRO_Package_Upload_Info $uploadInfo): void {
            if ($uploadInfo->isDownloadFromRemote()) {
                return;
            }

            EmailSummary::getInstance()->addCancelledUpload($uploadInfo);
        });

        add_action('duplicator_pro_upload_complete', function (DUP_PRO_Package_Upload_Info $uploadInfo): void {
            if ($uploadInfo->isDownloadFromRemote()) {
                return;
            }

            EmailSummary::getInstance()->addSuccessfulUpload($uploadInfo);
        });
    }

    /**
     * Init cron on activation
     *
     * @return void
     */
    public static function activationAction(): void
    {
        $frequency = DUP_PRO_Global_Entity::getInstance()->getEmailSummaryFrequency();
        if ($frequency === EmailSummary::SEND_FREQ_NEVER) {
            return;
        }

        if (self::updateCron($frequency) == false) {
            DUP_PRO_Log::trace("FAILED TO INIT EMAIL SUMMARY CRON. Frequency: {$frequency}");
        }
    }

    /**
     * Removes cron on deactivation
     *
     * @return void
     */
    public static function deactivationAction(): void
    {
        if (self::updateCron(EmailSummary::SEND_FREQ_NEVER) == false) {
            DUP_PRO_Log::trace("FAILED TO REMOVE EMAIL SUMMARY CRON.");
        }
    }

    /**
     * Updates the WP Cron job base on frequency or settings
     *
     * @param string $frequency The frequency
     *
     * @return bool True if the cron was updated or false on error
     */
    private static function updateCron($frequency = '')
    {
        if (strlen($frequency) === 0) {
            $frequency = DUP_PRO_Global_Entity::getInstance()->getEmailSummaryFrequency();
        }

        if ($frequency === EmailSummary::SEND_FREQ_NEVER) {
            if (wp_next_scheduled(self::CRON_HOOK)) {
                //have to check return like this because
                //wp_clear_scheduled_hook returns void in WP < 5.1
                return !self::isFalseOrWpError(wp_clear_scheduled_hook(self::CRON_HOOK));
            } else {
                return true;
            }
        } else {
            if (wp_next_scheduled(self::CRON_HOOK) && self::isFalseOrWpError(wp_clear_scheduled_hook(self::CRON_HOOK))) {
                return false;
            }

            return !self::isFalseOrWpError(
                wp_schedule_event(
                    self::getFirstRunTime($frequency),
                    self::getCronSchedule($frequency),
                    self::CRON_HOOK
                )
            );
        }
    }

    /**
     * Update next send time on frequency setting change
     *
     * @param string $oldFrequency The old frequency
     * @param string $newFrequency The new frequency
     *
     * @return bool True if the cron was updated or false on error
     */
    public static function updateFrequency($oldFrequency, $newFrequency)
    {
        if ($oldFrequency === $newFrequency) {
            return true;
        }

        return self::updateCron($newFrequency);
    }

    /**
     * Get the cron schedule
     *
     * @param string $frequency The frequency
     *
     * @return string
     */
    private static function getCronSchedule($frequency): string
    {
        switch ($frequency) {
            case EmailSummary::SEND_FREQ_DAILY:
                return CronUtils::INTERVAL_DAILY;
            case EmailSummary::SEND_FREQ_WEEKLY:
                return CronUtils::INTERVAL_WEEKLY;
            case EmailSummary::SEND_FREQ_MONTHLY:
                return CronUtils::INTERVAL_MONTHLY;
            default:
                throw new Exception("Unknown frequency: " . $frequency);
        }
    }

    /**
     * Set next send time based on frequency
     *
     * @param string $frequency Frequency
     *
     * @return int
     */
    private static function getFirstRunTime($frequency)
    {
        switch ($frequency) {
            case EmailSummary::SEND_FREQ_DAILY:
                $firstRunTime = strtotime('tomorrow 14:00');
                break;
            case EmailSummary::SEND_FREQ_WEEKLY:
                $firstRunTime = strtotime('next monday 14:00');
                break;
            case EmailSummary::SEND_FREQ_MONTHLY:
                $firstRunTime = strtotime('first day of next month 14:00');
                break;
            case EmailSummary::SEND_FREQ_NEVER:
                return 0;
            default:
                throw new Exception("Unknown frequency: " . $frequency);
        }

        return $firstRunTime - SnapWP::getGMTOffset();
    }

    /**
     * Send email
     *
     * @return void
     */
    public static function send(): void
    {
        DUP_PRO_Log::trace("CRON: Sending email summary");
        $recipients = DUP_PRO_Global_Entity::getInstance()->getEmailSummaryRecipients();
        $frequency  = DUP_PRO_Global_Entity::getInstance()->getEmailSummaryFrequency();
        if (count($recipients) === 0 || $frequency === EmailSummary::SEND_FREQ_NEVER) {
            DUP_PRO_Log::trace("CRON: No recipients or frequency is never");
            return;
        }

        $parsedHomeUrl = wp_parse_url(home_url());
        $siteDomain    = ($parsedHomeUrl['host'] ?? '');

        if (is_multisite() && isset($parsedHomeUrl['path'])) {
            $siteDomain .= $parsedHomeUrl['path'];
        }

        $subject = sprintf(
            esc_html_x(
                'Your Duplicator Summary for %s',
                '%s is the site domain',
                'duplicator-pro'
            ),
            $siteDomain
        );

        $content = TplMng::getInstance()->render('mail/email_summary', EmailSummary::getInstance()->getData(), false);

        add_filter('wp_mail_content_type', [self::class, 'getMailContentType']);
        if (!wp_mail($recipients, $subject, $content)) {
            DUP_PRO_Log::trace("FAILED TO SEND EMAIL SUMMARY.");
            DUP_PRO_Log::traceObject("RECIPIENTS: ", $recipients);
            return;
        } else {
            DUP_PRO_Log::trace("EMAIL SUMMARY SENT SUCCESSFULLY.");
        }

        EmailSummary::getInstance()->reset();
    }

    /**
     * Get mail content type
     *
     * @return string
     */
    public static function getMailContentType(): string
    {
        return 'text/html';
    }

    /**
     * Returns true if is false or wp_error
     *
     * @param mixed $value The value
     *
     * @return bool
     */
    private static function isFalseOrWpError($value): bool
    {
        return $value === false || is_wp_error($value);
    }
}
