<?php
/*
Plugin Name: OTP Registration and Login
Description: Adds OTP functionality to the registration and login forms.
Version: 1.2.7
Author: Suncode
*/
include_once "woo_edit_page_addon.php";
include_once "woo_edit_page_addon.php";
include_once "compulsion_mobile_number.php";
include_once "inc/admin/admin_functions.php"; //v13.2
include_once "providers/providers_list.php"; //v13.2

// Function to create the table if it doesn't exist
function create_sc_otp_table_if_not_exists() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'sc_otp_tbl';

    // Check if the table exists
    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
        $charset_collate = $wpdb->get_charset_collate();

        $sql = "CREATE TABLE $table_name (
            id mediumint(9) NOT NULL AUTO_INCREMENT,
            user_id mediumint(9) DEFAULT NULL,
            mobile_number varchar(15) NOT NULL,
            verify_code varchar(10) NOT NULL,
            code_expiration_time datetime NOT NULL,
            PRIMARY KEY  (id)
        ) $charset_collate;";

        require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
        dbDelta($sql);
    }
}

// Create the table if it doesn't exist when the plugin loads
add_action('plugins_loaded', 'create_sc_otp_table_if_not_exists');

// Modified function to save the OTP and update when it already exists
function sc_save_otp_to_db($mobile_number, $verify_code) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'sc_otp_tbl';
    $expiration_time = date('Y-m-d H:i:s', strtotime('+2 minutes')); // Set expiration time for 2 minutes

    // Check if the mobile number already exists in the database
    $existing_record = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE mobile_number = %s", $mobile_number));

    if ($existing_record) {
        // Update the existing record
        $wpdb->update($table_name, array(
            'verify_code' => $verify_code,
            'code_expiration_time' => $expiration_time
        ), array('mobile_number' => $mobile_number));
    } else {
        // Insert new record
        $wpdb->insert($table_name, array(
            'mobile_number' => $mobile_number,
            'verify_code' => $verify_code,
            'code_expiration_time' => $expiration_time,
        ));
    }
}

// Function to determine redirect URL based on Redux settings
function sc_get_redirect_url() {
    // Default to home URL if WooCommerce or settings are unavailable
    $account_link = home_url('/');

    // Check if WooCommerce is active and settings are available
    if (function_exists('wc_get_page_id')) {
        $myaccount_page_id = wc_get_page_id('myaccount');
        if ($myaccount_page_id && get_post($myaccount_page_id)) {
            $account_link = get_permalink($myaccount_page_id);
        }
    }

    $redirect = $account_link; // Default redirect to account page or home

    if (class_exists('Redux')) {
        $redirect_type = codebean_option('login_redirect_type') ?: '';
        if ($redirect_type === 'current') {
            // Use wp_get_referer for AJAX requests to get the originating page
            $referer = wp_get_referer();
            $redirect = $referer && strpos($referer, 'admin-ajax.php') === false ? esc_url($referer) : $account_link;
        } elseif ($redirect_type === 'custom') {
            // Redirect to custom URL
            $custom_url = codebean_option('login_custom_redirect_url');
            $redirect = !empty($custom_url) ? esc_url($custom_url) : $account_link;
        }
    }

    return $redirect;
}

// Function to check the OTP upon registration
function sc_check_otp_and_register($mobile_number, $entered_code) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'sc_otp_tbl';

    // Fetch latest OTP details
    $otp_info = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE mobile_number = %s ORDER BY id DESC LIMIT 1", $mobile_number));

    if ($otp_info) {
        // Check if the code matches and ensure it's not expired
        if ($otp_info->verify_code === $entered_code && new DateTime() < new DateTime($otp_info->code_expiration_time)) {
            // Check if user already exists
            $user = get_user_by('login', $mobile_number); 
            if (!$user) {
                // Register the user if they do not exist
                $user_id = wp_create_user($mobile_number, wp_generate_password(), $mobile_number . '@example.com');
                if (!is_wp_error($user_id)) {
                    // Update user_id in the OTP table
                    $wpdb->update($table_name, array('user_id' => $user_id), array('id' => $otp_info->id));
                    update_user_meta($user_id, 'user_phone', $mobile_number); // Save phone number v13.2
                    update_user_meta($user_id, 'billing_phone', $mobile_number); // Save phone number v13.2
                    // Set display name based on Redux settings
                    $display_name_type = 'mobile'; // Default
                    if (class_exists('Redux')) {
                        $display_name_type = codebean_option('new_user_display_name_type');
                    }
                    $display_name = $mobile_number; // Default to mobile number
                    if ($display_name_type === 'custom') {
                        $custom_display_name = __('New User', 'studiare-core'); // Default
                        if (class_exists('Redux')) {
                            $custom_display_name = codebean_option('new_user_custom_display_name');
                        }
                        $display_name = !empty($custom_display_name) ? $custom_display_name : __('New User', 'studiare-core');
                    }

                    // Update user display name
                    wp_update_user(array(
                        'ID' => $user_id,
                        'display_name' => $display_name
                    ));

                    // Log the user in
                    wp_set_auth_cookie($user_id);
                    return array('status' => 'success', 'message' => __('Registration successful!', 'studiare-core'));
                } else {
                    return array('status' => 'error', 'message' => __('Failed to register user.', 'studiare-core'));
                }
            } else {
                // Log the user in if they already exist
                wp_set_auth_cookie($user->ID);
                return array('status' => 'success', 'message' => __('User already exists. You are logged in!', 'studiare-core'));
            }
        } else {
            return array('status' => 'error', 'message' => __('Invalid or expired OTP', 'studiare-core'));
        }
    } else {
        return array('status' => 'error', 'message' => __('No OTP found for this mobile number', 'studiare-core')); 
    }
}

// Function to validate phone number
function sc_validate_phone_number($phone) {
    // Remove country code
    $phone = preg_replace('/^\+\d{1,3}/', '', $phone); // Strip country code
    $phone = ltrim($phone, '0'); // Remove leading zeros

    // Validate phone number must not start with '0' now
    return (strlen($phone) >= 10 && !preg_match('/^0/', $phone)) ? $phone : false;
}

// Enqueue scripts and styles
function otp_registration_enqueue_scripts() {
    //wp_enqueue_script('otp-registration-scripts', plugin_dir_url(__FILE__) . 'js/otp-registration-scripts.js', array('jquery'), '1.2', true);
    wp_enqueue_script('otp-registration-scripts', plugin_dir_url(__FILE__) . 'js/otp-registration-scripts.js', ['jquery'], '1.2', true);
    wp_localize_script('otp-registration-scripts', 'otp_ajax_object', [
        'ajax_url' => admin_url('admin-ajax.php'),
        'current_url' => esc_url(home_url(add_query_arg([]))),
        'nonce' => wp_create_nonce('otp_send_verification_nonce')
    ]);
}
add_action('wp_enqueue_scripts', 'otp_registration_enqueue_scripts');

// Register shortcode for the registration form
function suncode_otp_registration_form_shortcode() {
    if (is_user_logged_in()) {
        return;
    }
    ob_start();
    ?>
    <style>
    @keyframes sc_otp_spinner {
      to {transform: rotate(360deg);}
    }
     
    .sc_otp_spinner:before {
      content: '';
      box-sizing: border-box;
      position: absolute;
      top: 50%;
      left: 50%;
      width: 20px;
      height: 20px;
      margin-top: -10px;
      margin-left: -10px;
      border-radius: 50%;
      border: 1px solid #f6f;
      border-top-color: #0e0;
      border-right-color: #0dd;
      border-bottom-color: #f90;
      animation: sc_otp_spinner .6s linear infinite;
    }
    div#sc_otp_reg_message, div#sc_otp_login_message {
        position: relative;
        min-height: 40px;
        align-content: center;
    }
    .sc_otp_btn { margin: 10px 0; }
    input#otp_phone, input#otp_reg_phone { letter-spacing: 5px; }
    input#otp_phone::placeholder, input#otp_reg_phone::placeholder { letter-spacing: 0px; }
    button.button.sc_otp_btn.sc_cmn_timer_on { background: gainsboro; }
    .success { color: green; }
    .error { color: red; }
    </style>
    <!-- Registration Form -->
    <form id="otp-registration-form" method="post" class="woocommerce-form">
        <div>
            <label for="otp_reg_phone"><?php echo __('Phone Number', 'studiare-core'); ?>:</label>
            <input type="text" name="otp_reg_phone" id="otp_reg_phone" placeholder="<?php echo __('Mobile Number', 'studiare-core'); ?>" required>
        </div>
        <div id="otp-container" style="display: none;">
            <label for="otp_reg"><?php echo __('OTP', 'studiare-core'); ?>:</label>
            <input type="text" name="otp_reg" id="otp_reg" required>
        </div>
        <div>
            <button class="button sc_otp_btn" type="button" id="send-otp" data-mtitle="<?php echo __('Send Verification Code', 'studiare-core'); ?>" data-countertxt="<?php echo __('Seconds', 'studiare-core'); ?>"><?php echo __('Send Verification Code', 'studiare-core'); ?></button>
            <button class="button sc_otp_btn" type="submit" id="register" style="display: none;"><?php echo __('Register', 'studiare-core'); ?></button>
            <div class="sc_otp_reg_counter"></div>
        </div>
        <div id="sc_otp_reg_message"></div>
    </form>
    <?php
    return ob_get_clean();
}
add_shortcode('suncode_otp_registration_form', 'suncode_otp_registration_form_shortcode');

// Register shortcode for the login form
function suncode_otp_login_form_shortcode() {
    if (is_user_logged_in()) {
        return;
    }
    ob_start();
    ?>
    <style>
    @keyframes sc_otp_spinner {
      to {transform: rotate(360deg);}
    }
     
    .sc_otp_spinner:before {
      content: '';
      box-sizing: box-box;
      position: absolute;
      top: 50%;
      left: 50%;
      width: 20px;
      height: 20px;
      margin-top: -10px;
      margin-left: -10px;
      border-radius: 50%;
      border: 1px solid #f6f;
      border-top-color: #0e0;
      border-right-color: #0dd;
      border-bottom-color: #f90;
      animation: sc_otp_spinner .6s linear infinite;
    }
    button.button.sc_otp_btn.sc_cmn_timer_on { background: gainsboro; }
    .success { color: green; }
    .error { color: red; }
    </style>
    <!-- Login Form -->
    <form id="otp-login-form" method="post" class="woocommerce-form">
        <div>
            <label for="otp_phone"><?php echo __('Phone Number', 'studiare-core'); ?>:</label>
            <input type="text" name="otp_phone" id="otp_phone" placeholder="<?php echo __('Mobile Number', 'studiare-core'); ?>" required>
        </div>
        <div id="otp-login-container" style="display: none;">
            <label for="otp_back"><?php echo __('OTP', 'studiare-core'); ?>:</label>
            <input type="text" name="otp_back" id="otp_back" required>
        </div>
        <div>
            <button class="button sc_otp_btn" type="button" id="send-login-otp" data-mtitle="<?php echo __('Send OTP Code', 'studiare-core'); ?>" data-countertxt="<?php echo __('Seconds', 'studiare-core'); ?>"><?php echo __('Send OTP Code', 'studiare-core'); ?></button>
            <button class="button sc_otp_btn" type="submit" id="login" style="display: none;"><?php echo __('Login', 'studiare-core'); ?></button>
        </div>
        <div id="sc_otp_login_message"></div>
    </form>
    <?php
    return ob_get_clean();
}
add_shortcode('suncode_otp_login_form', 'suncode_otp_login_form_shortcode');

// AJAX callback to send verification code
function otp_send_verification_code() {
    //$phone = $_POST['otp_reg_phone'] ?? $_POST['otp_phone']; // Grab the phone from either form
    
    check_ajax_referer('otp_send_verification_nonce', 'nonce'); // Add nonce check
    //check_ajax_referer('otp_ajax_object', 'nonce'); // Add nonce check
    $phone = sanitize_text_field($_POST['otp_reg_phone'] ?? $_POST['otp_phone']);

    // Validate phone before proceeding
    $validated_phone = sc_validate_phone_number($phone);
    if (!$validated_phone) {
        wp_send_json_error(['message' => __('Invalid phone number.', 'studiare-core')]);
        return;
    }

    // Check if the phone number already exists for registration
    if (isset($_POST['otp_reg_phone'])) {
        $user = get_user_by('login', $validated_phone);
        if ($user) {
            wp_send_json_error(array('message' => __('This number already exists.', 'studiare-core')));
        }
    }
    
    // Check if the phone number already exists for login
    if (isset($_POST['otp_phone'])) {
        $user = get_user_by('login', $validated_phone);
        if (!$user) {
            wp_send_json_error(array('message' => __('This number does not exist.', 'studiare-core')));
        }
    }

    // Call your SMS provider's API to send the verification code
    $apiResponse = sc_call_sms_provider_api($validated_phone);
    $apiResponse = json_decode($apiResponse, true);

    if ($apiResponse['status'] === 'ارسال موفق بود' || $apiResponse['status'] === 'OK' || $apiResponse['status'] === 1){
        // Save the OTP code and expiration time in the database
        $otp = $apiResponse['code'];
        sc_save_otp_to_db($validated_phone, $otp);
        wp_send_json_success(array('message' => __('Verification code sent successfully.', 'studiare-core')));
    } else {
        //wp_send_json_error(array('message' => __('Failed to send verification code.', 'studiare-core')));
        wp_send_json_error(array('message' => $apiResponse));
    }
}
add_action('wp_ajax_otp_send_verification_code', 'otp_send_verification_code');
add_action('wp_ajax_nopriv_otp_send_verification_code', 'otp_send_verification_code');

// AJAX callback to validate OTP for registration
function otp_validate_otp() {
    $otp = $_POST['otp_reg'];
    $phone = $_POST['otp_reg_phone'];

    // Validate the phone number before proceeding
    $validated_phone = sc_validate_phone_number($phone);
    if (!$validated_phone) {
        wp_send_json_error(array('message' => __('Invalid phone number.', 'studiare-core')));
        return;
    }

    // Validate OTP and register if valid
    $result = sc_check_otp_and_register($validated_phone, $otp);
    if ($result['status'] === 'success') {
        // Redirect on successful registration
        wp_send_json_success(array('message' => $result['message'], 'redirect_url' => sc_get_redirect_url()));
    } else {
        wp_send_json_error(array('message' => $result['message']));
    }
}
add_action('wp_ajax_otp_validate_otp', 'otp_validate_otp');
add_action('wp_ajax_nopriv_otp_validate_otp', 'otp_validate_otp');

// AJAX callback to validate OTP for login
function otp_login_validate_otp() {
    $otp = sanitize_text_field($_POST['otp_back']);
    $phone = sanitize_text_field($_POST['otp_phone']);
    $validated_phone = sc_validate_phone_number($phone);
    if (!$validated_phone) {
        wp_send_json_error(['message' => __('Invalid phone number.', 'studiare-core')]);
        return;
    }

    global $wpdb;
    $table_name = $wpdb->prefix . 'sc_otp_tbl';
    $otp_info = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE mobile_number = %s ORDER BY id DESC LIMIT 1", $validated_phone));

    if ($otp_info && $otp_info->verify_code === $otp && new DateTime() < new DateTime($otp_info->code_expiration_time)) {
        $uid = studi_get_user_by_vali_number($validated_phone);
        $user = $uid ? get_user_by('id', $uid) : false;
        if ($user) {
            wp_set_auth_cookie($user->ID);
            wp_send_json_success(['message' => __('Login successful!', 'studiare-core'), 'redirect_url' => sc_get_redirect_url()]);
        } else {
            wp_send_json_error(['message' => __('User not found.', 'studiare-core')]);
        }
    } else {
        wp_send_json_error(['message' => __('Invalid or expired OTP.', 'studiare-core')]);
    }
}
add_action('wp_ajax_otp_login_validate_otp', 'otp_login_validate_otp');
add_action('wp_ajax_nopriv_otp_login_validate_otp', 'otp_login_validate_otp');

// Function to delete OTP entry when user is deleted
function sc_delete_otp_on_user_delete($user_id) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'sc_otp_tbl';

    // Get the user's mobile number
    $user = get_userdata($user_id);
    if ($user && $user->user_login) { // Assuming user_login holds the phone number
        $mobile_number = $user->user_login; // Adjust if necessary
        $wpdb->delete($table_name, array('mobile_number' => $mobile_number));
    }
}
add_action('delete_user', 'sc_delete_otp_on_user_delete');

// Function to call SMS provider API
function MMsc_call_sms_provider_api($phone) {
    // Default values
    $otp_provider = 'mellipayamak';
    $mellipayamak_api_address = '';
    $suncode_api_address = '';

    // Fetch all Redux settings at once if Redux is available
    if (class_exists('Redux')) {
        $otp_provider = codebean_option('otp_provider') ?: $otp_provider;
        $mellipayamak_api_address = codebean_option('mellipayamak_api_address') ?: $mellipayamak_api_address;
        $suncode_api_address = codebean_option('suncode_api_address') ?: $suncode_api_address;
    }

    // Melli Payamak
    if ($otp_provider == 'mellipayamak') {
        $url = $mellipayamak_api_address;
        $data = array('to' => $phone);
        $data_string = json_encode($data);
        $ch = curl_init($url);                          
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");                      
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json', 'Content-Length: ' . strlen($data_string)));
        $result = curl_exec($ch);
        curl_close($ch);
        return $result;
    }

    // Suncode
    if ($otp_provider == 'suncode') {
        $url = $suncode_api_address;
        $data = array('to' => $phone);
        $data_string = json_encode($data);
        $ch = curl_init($url);                          
        curl_setopt($ch, CURLOPT_CUSTOMREQUEST, "POST");                      
        curl_setopt($ch, CURLOPT_POSTFIELDS, $data_string);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Content-Type: application/json', 'Content-Length: ' . strlen($data_string)));
        $result = curl_exec($ch);
        curl_close($ch);
        return $result;
    }

    // Return empty response if no provider is matched
    return json_encode(array('status' => 'error', 'message' => 'No SMS provider configured.'));
}
?>